# Check-In Completed Event

This document describes the implementation of the check-in completed event in the Weinanlieferung Bundle.

## Overview

The check-in completed event is triggered when a reservation check-in is completed. This event allows other parts of the application to react to check-ins, such as sending notifications, updating statistics, or performing other actions.

## Implementation Details

### Files Created/Modified

1. **Created: `src/Event/CheckInCompletedEvent.php`**
   - Defines the event class that is dispatched when a check-in is completed
   - Contains the reservation data and model
   - Provides getter methods to access the data

2. **Modified: `src/Controller/Frontend/Ajax/SlotAjaxController.php`**
   - Added the event dispatcher as a dependency
   - Updated the constructor to inject the event dispatcher
   - Modified the `updateCheckin()` method to dispatch the event after a successful check-in

3. **Created: `src/EventListener/CheckInCompletedListener.php`**
   - Example listener that demonstrates how to subscribe to the check-in completed event
   - Logs when a check-in is completed

## How to Use the Event

### Listening to the Event

To listen to the check-in completed event, create a class that implements `EventSubscriberInterface`:

```php
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use vonRotenberg\WeinanlieferungBundle\Event\CheckInCompletedEvent;

class YourListener implements EventSubscriberInterface
{
    public static function getSubscribedEvents()
    {
        return [
            CheckInCompletedEvent::NAME => 'onCheckInCompleted',
        ];
    }

    public function onCheckInCompleted(CheckInCompletedEvent $event)
    {
        // Get the reservation data
        $reservationData = $event->getReservationData();
        $reservationModel = $event->getReservationModel();

        // Perform your custom actions here
    }
}
```

### Available Data

The event provides access to:

1. **Reservation Data Array**: Contains all fields from the reservation record
   ```php
   $reservationData = $event->getReservationData();
   $id = $reservationData['id'];
   $checkedIn = $reservationData['checked_in'];
   $checkedInOn = $reservationData['checked_in_on'];
   // etc.
   ```

2. **Reservation Model**: Provides the Contao model object for the reservation
   ```php
   $reservationModel = $event->getReservationModel();
   $id = $reservationModel->id;
   $slot = $reservationModel->getRelated('pid');
   // etc.
   ```

## Testing

The example listener `CheckInCompletedListener` logs when a check-in is completed. You can check the Contao system log to verify that the event is being dispatched correctly.

When a check-in is completed, you should see a log entry like:
```
Check-in completed for reservation ID: 123
```

## Conclusion

This implementation provides a flexible way to react to check-in events in the Weinanlieferung Bundle. By using Symfony's event system, it allows for loose coupling between the check-in process and any additional functionality that needs to be triggered when a check-in occurs.
