<?php

/**
 * OBG Customizations
 *
 * Copyright (c) 2021 vonRotenberg
 *
 * @license commercial
 */

namespace vonRotenberg\RealEstateListingBundle\Model;

use Contao\CoreBundle\File\ModelMetadataTrait;
use Contao\Date;
use Contao\Model;
use Contao\Model\Collection;

/**
 * Reads and writes assets
 *
 * @property string|integer         $id
 * @property string|integer         $pid
 * @property string|integer         $tstamp
 * @property string                 $title
 * @property string                 $address
 * @property string                 $assetNo
 * @property string|float           $livingSpace
 * @property string|integer         $numberOfRooms
 * @property string|float           $coldRent
 * @property string|float           $advanceOperatingCosts
 * @property string                 $condition
 * @property string                 $constructionYear
 * @property string|integer|null    $cooperativeShares
 * @property string|null            $features
 * @property string|                $heatingType
 * @property string|float           $energyConsumption
 * @property string|null            $gallerySRC
 * @property string|null            $orderSRC
 * @property string|null            $floorPlansSRC
 * @property string|null            $floorPlansOrderSRC
 * @property string                 $availability
 * @property string|integer         $availableFrom
 * @property string|null            $description
 * @property string|boolean         $published
 * @property string|integer         $start
 * @property string|integer         $stop
 *
 * @method static RealEstateAssetsModel|null findById($id, array $opt=array())
 * @method static RealEstateAssetsModel|null findByPk($id, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneBy($col, $val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByPid($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByTstamp($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByTitle($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByAddress($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByAssetNo($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByLivingSpace($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByNumberOfRooms($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByColdRent($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByAdvanceOperatingCosts($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByCondition($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByConstructionYear($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByCooperativeShares($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByFeatures($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByHeatingType($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByEnergyConsumption($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByGallerySRC($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByFloorPlansSRC($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByAvailability($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByAvailableFrom($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByDescription($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByPublished($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByStart($val, array $opt=array())
 * @method static RealEstateAssetsModel|null findOneByStop($val, array $opt=array())
 *
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByPid($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByTstamp($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByTitle($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByAddress($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByAssetNo($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByLivingSpace($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByNumberOfRooms($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByColdRent($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByAdvanceOperatingCosts($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByCondition($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByConstructionYear($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByCooperativeShares($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByFeatures($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByHeatingType($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByEnergyConsumption($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByGallerySRC($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByFloorPlansSRC($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByAvailability($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByAvailableFrom($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByDescription($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByPublished($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByStart($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findByStop($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findMultipleByIds($val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findBy($col, $val, array $opt=array())
 * @method static Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null findAll(array $opt=array())
 *
 * @method static integer countById($id, array $opt=array())
 * @method static integer countByPid($val, array $opt=array())
 * @method static integer countByTstamp($val, array $opt=array())
 * @method static integer countByTitle($val, array $opt=array())
 * @method static integer countByAddress($val, array $opt=array())
 * @method static integer countByAssetNo($val, array $opt=array())
 * @method static integer countByLivingSpace($val, array $opt=array())
 * @method static integer countByNumberOfRooms($val, array $opt=array())
 * @method static integer countByColdRent($val, array $opt=array())
 * @method static integer countByAdvanceOperatingCosts($val, array $opt=array())
 * @method static integer countByCondition($val, array $opt=array())
 * @method static integer countByConstructionYear($val, array $opt=array())
 * @method static integer countByCooperativeShares($val, array $opt=array())
 * @method static integer countByFeatures($val, array $opt=array())
 * @method static integer countByHeatingType($val, array $opt=array())
 * @method static integer countByEnergyConsumption($val, array $opt=array())
 * @method static integer countByGallerySRC($val, array $opt=array())
 * @method static integer countByFloorPlansSRC($val, array $opt=array())
 * @method static integer countByAvailability($val, array $opt=array())
 * @method static integer countByAvailableFrom($val, array $opt=array())
 * @method static integer countByDescription($val, array $opt=array())
 * @method static integer countByPublished($val, array $opt=array())
 * @method static integer countByStart($val, array $opt=array())
 * @method static integer countByStop($val, array $opt=array())
 *
 * @author Benjamin Roth <https://www.vonrotenberg.de>
 */
class RealEstateAssetsModel extends Model
{

	/**
	 * Table name
	 * @var string
	 */
	protected static $strTable = 'tl_vr_real_estate_assets';

  /**
   * Find a published assets from one or more categories
   *
   * @param mixed $varId      The numeric id or alias name
   * @param array $arrOptions An optional options array
   *
   * @return RealEstateAssetsModel|null A model or null if there is no asset
   */
  public static function findPublishedByIdOrAlias($varId, array $arrOptions=array())
  {
    $t = static::$strTable;
    $arrColumns = !preg_match('/^[1-9]\d*$/', $varId) ? array("BINARY $t.alias=?") : array("$t.id=?");

    if (!static::isPreviewMode($arrOptions))
    {
      $time = Date::floorToMinute();
      $arrColumns[] = "$t.published='1' AND ($t.start='' OR $t.start<='$time') AND ($t.stop='' OR $t.stop>'$time')";
    }

    return static::findOneBy($arrColumns, $varId, $arrOptions);
  }

	/**
	 * Find a published assets from one or more categories
	 *
	 * @param array   $arrPids    An array of parent IDs
   * @param integer $intLimit   An optional limit
   * @param integer $intOffset  An optional offset
   * @param array   $arrOptions An optional options array
	 *
	 * @return Collection|RealEstateAssetsModel[]|RealEstateAssetsModel|null A collection of models or null if there are no assets
	 */
	public static function findPublishedByParent($arrPids, $intLimit=0, $intOffset=0, array $arrOptions=array())
	{
		if (empty($arrPids) || !\is_array($arrPids))
		{
			return null;
		}

		$t = static::$strTable;
		$arrColumns = ["$t.pid IN(" . implode(',', array_map('\intval', $arrPids)) . ")"];

    if (!static::isPreviewMode($arrOptions))
    {
      $time = Date::floorToMinute();
      $arrColumns[] = "$t.published='1' AND ($t.start='' OR $t.start<='$time') AND ($t.stop='' OR $t.stop>'$time')";
    }

    if (!isset($arrOptions['order']))
    {
      $arrOptions['order']  = "FIELD($t.pid,".implode(',',$arrPids)."), $t.title ASC, $t.livingSpace ASC";
    }

    $arrOptions['limit']  = $intLimit;
    $arrOptions['offset'] = $intOffset;

    return static::findBy($arrColumns, null, $arrOptions);
	}

  /**
   * Find a published assets from one or more categories
   *
   * @param array   $arrPids    An array of parent IDs
   * @param array   $arrOptions An optional options array
   *
   * @return integer The number of matching rows
   */
  public static function countPublishedByParent($arrPids, array $arrOptions=array())
  {
    if (empty($arrPids) || !\is_array($arrPids))
    {
      return 0;
    }

    $t = static::$strTable;
    $arrColumns = ["$t.pid IN(" . implode(',', array_map('\intval', $arrPids)) . ")"];

    if (!static::isPreviewMode($arrOptions))
    {
      $time = Date::floorToMinute();
      $arrColumns[] = "$t.published='1' AND ($t.start='' OR $t.start<='$time') AND ($t.stop='' OR $t.stop>'$time')";
    }

    return static::countBy($arrColumns, null, $arrOptions);
  }
}
