<?php

/**
 * OBG Customizations
 *
 * Copyright (c) 2021 vonRotenberg
 *
 * @license commercial
 */

namespace vonRotenberg\RealEstateListingBundle\Model;

use Contao\Date;
use Contao\Model;
use Contao\Model\Collection;

/**
 * Reads and writes assets
 *
 * @property string|integer         $id
 * @property string|integer         $tstamp
 * @property string                 $address
 * @property string                 $postal
 * @property string                 $city
 * @property string|null            $gallerySRC
 * @property string|null            $orderSRC
 * @property string|null            $description
 * @property string|boolean         $published
 * @property string|integer         $start
 * @property string|integer         $stop
 *
 * @method static ManagedPropertyModel|null findById($id, array $opt = array())
 * @method static ManagedPropertyModel|null findByPk($id, array $opt = array())
 * @method static ManagedPropertyModel|null findOneBy($col, $val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByTstamp($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByAddress($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByPostal($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByCity($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByGallerySRC($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByDescription($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByPublished($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByStart($val, array $opt = array())
 * @method static ManagedPropertyModel|null findOneByStop($val, array $opt = array())
 *
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByTstamp($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByAddress($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByPostal($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByCity($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByGallerySRC($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByDescription($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByPublished($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByStart($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findByStop($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findMultipleByIds($val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findBy($col, $val, array $opt = array())
 * @method static Collection|ManagedPropertyModel[]|ManagedPropertyModel|null findAll(array $opt = array())
 *
 * @method static integer countById($id, array $opt = array())
 * @method static integer countByTstamp($val, array $opt = array())
 * @method static integer countByAddress($val, array $opt = array())
 * @method static integer countByPostal($val, array $opt = array())
 * @method static integer countByCity($val, array $opt = array())
 * @method static integer countByGallerySRC($val, array $opt = array())
 * @method static integer countByDescription($val, array $opt = array())
 * @method static integer countByPublished($val, array $opt = array())
 * @method static integer countByStart($val, array $opt = array())
 * @method static integer countByStop($val, array $opt = array())
 *
 * @author Benjamin Roth <https://www.vonrotenberg.de>
 */
class ManagedPropertyModel extends Model
{

    /**
     * Table name
     * @var string
     */
    protected static $strTable = 'tl_vr_re_managedProperties';

    /**
     * Find a published assets from one or more categories
     *
     * @param mixed $varId      The numeric id or alias name
     * @param array $arrOptions An optional options array
     *
     * @return ManagedPropertyModel|null A model or null if there is no asset
     */
    public static function findPublishedByIdOrAlias($varId, array $arrOptions = array())
    {
        $t = static::$strTable;
        $arrColumns = !preg_match('/^[1-9]\d*$/', $varId) ? array("BINARY $t.alias=?") : array("$t.id=?");

        if (!static::isPreviewMode($arrOptions))
        {
            $time = Date::floorToMinute();
            $arrColumns[] = "$t.published='1' AND ($t.start='' OR $t.start<='$time') AND ($t.stop='' OR $t.stop>'$time')";
        }

        return static::findOneBy($arrColumns, $varId, $arrOptions);
    }

    /**
     * Find a published assets from one or more categories
     *
     * @param array   $arrPids    An array of parent IDs
     * @param integer $intLimit   An optional limit
     * @param integer $intOffset  An optional offset
     * @param array   $arrOptions An optional options array
     *
     * @return Collection|ManagedPropertyModel[]|ManagedPropertyModel|null A collection of models or null if there are no assets
     */
    public static function findAllPublished($intLimit = 0, $intOffset = 0, array $arrOptions = array())
    {
        $t = static::$strTable;
        $arrColumns = [];
        $arrValues = [];

        if (!static::isPreviewMode($arrOptions))
        {
            $time = Date::floorToMinute();
            $arrColumns[] = "$t.published='1' AND ($t.start='' OR $t.start<='$time') AND ($t.stop='' OR $t.stop>'$time')";
        }

        if (!isset($arrOptions['order']))
        {
            $arrOptions['order'] = "$t.city ASC, $t.livingSpace ASC";
        }

        $arrOptions['limit'] = $intLimit;
        $arrOptions['offset'] = $intOffset;

        if (isset($arrOptions['column']))
        {
            $arrColumns = array_merge($arrColumns,array_values($arrOptions['column']));
            unset($arrOptions['column']);
        }
        if (isset($arrOptions['value']))
        {
            $arrValues = array_merge($arrValues,array_values($arrOptions['value']));
            unset($arrOptions['value']);
        }

        return static::findBy($arrColumns, $arrValues, $arrOptions);
    }

    /**
     * Find a published assets from one or more categories
     *
     * @param array $arrPids    An array of parent IDs
     * @param array $arrOptions An optional options array
     *
     * @return integer The number of matching rows
     */
    public static function countAllPublished(array $arrOptions = array())
    {
        $t = static::$strTable;
        $arrColumns = [];
        $arrValues = [];

        if (!static::isPreviewMode($arrOptions))
        {
            $time = Date::floorToMinute();
            $arrColumns[] = "$t.published='1' AND ($t.start='' OR $t.start<='$time') AND ($t.stop='' OR $t.stop>'$time')";
        }

        if (isset($arrOptions['column']))
        {
            $arrColumns = array_merge($arrColumns,array_values($arrOptions['column']));
            unset($arrOptions['column']);
        }
        if (isset($arrOptions['value']))
        {
            $arrValues = array_merge($arrValues,array_values($arrOptions['value']));
            unset($arrOptions['value']);
        }
        return static::countBy($arrColumns, $arrValues, $arrOptions);
    }
}
