import { LitElement, html, css } from 'lit';

export class WinzerDayView extends LitElement {
    static properties = { events: { type: Array } };
    static styles = css`
        :host { display: block; background: var(--bg-white); border-radius: var(--radius-card); box-shadow: var(--shadow-sm); overflow: hidden; height: 600px; display: flex; flex-direction: column; }
        .scroll-container { flex: 1; overflow-y: auto; position: relative; }
        .day-grid { display: grid; grid-template-columns: 50px 1fr; grid-auto-rows: 20px; background-image: linear-gradient(to bottom, var(--border-color) 1px, transparent 1px); background-size: 100% 80px; background-attachment: local; }
        .time-label { grid-column: 1; text-align: right; padding-right: 10px; font-size: 0.75rem; color: var(--text-muted); line-height: 1; transform: translateY(-50%); margin-top: 1px; }
        .event-item { grid-column: 2; background-color: var(--primary-light); border-left: 4px solid var(--primary-color); border: 1px solid rgba(0,0,0,0.05); border-left-width: 4px; border-radius: 4px; padding: 6px 8px; font-size: 0.8rem; color: var(--primary-color); overflow: hidden; box-shadow: var(--shadow-sm); cursor: pointer; margin-right: 10px; margin-bottom: 2px; display: flex; flex-direction: column; justify-content: flex-start; gap: 4px; transition: transform 0.1s, filter 0.1s; }
        .event-item:active { transform: scale(0.98); filter: brightness(0.95); }
        .event-title { font-weight: 600; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; line-height: 1.2; }
        .event-details-row { display: flex; justify-content: space-between; align-items: center; font-size: 0.7rem; opacity: 0.8; line-height: 1.2; }
        .capacity-text { font-weight: 600; }
        .progress-track { width: 100%; height: 6px; background-color: rgba(255,255,255,0.6); border-radius: 3px; overflow: hidden; margin-top: auto; }
        .progress-fill { height: 100%; background-color: var(--primary-color); border-radius: 3px; transition: width 0.3s ease; }
        .event-item.fully-booked { border-color: var(--alert-color); border-left-width: 4px; background-color: var(--alert-light); color: #b71c1c; }
        .event-item.fully-booked .progress-fill { background-color: var(--alert-color); }
        .badge-booked-out { background-color: var(--alert-color); color: white; font-size: 0.6rem; padding: 2px 6px; border-radius: 4px; font-weight: bold; text-transform: uppercase; margin-right: 6px; display: inline-block; }
        .event-item.short-event { flex-direction: row; align-items: center; justify-content: space-between; padding: 0 6px; padding-bottom: 4px; gap: 8px; position: relative; }
        .event-item.short-event .event-title { flex: 1; min-width: 0; }
        .event-item.short-event .event-details-row { flex-shrink: 0; display: flex; align-items: center; }
        .event-item.short-event .progress-track { position: absolute; bottom: 0; left: 0; right: 0; width: 100%; height: 2px; margin-top: 0; border-radius: 0; background-color: rgba(0,0,0,0.05); }
        .event-item.short-event .progress-fill { border-radius: 0; }
        .grid-spacer { grid-column: 1 / -1; grid-row: 97; height: 20px; }
    `;
    _timeToGridRow(t) { if(!t) return 1; const [h, m] = t.split(':').map(Number); return (h * 4) + Math.floor(m / 15) + 1; }
    _formatNumber(n) { return new Intl.NumberFormat('de-DE').format(n); }
    render() {
        const hours = Array.from({ length: 24 }, (_, i) => i);
        return html`<div class="scroll-container"><div class="day-grid">${hours.map(h => html`<div class="time-label" style="grid-row: ${(h * 4) + 1}">${h.toString().padStart(2, '0')}:00</div>`)}${this.events ? this.events.map(ev => { const startRow = this._timeToGridRow(ev.start); const endRow = this._timeToGridRow(ev.end); const durationRows = endRow - startRow; const isShort = durationRows <= 2; const isFull = (ev.total && ev.booked >= ev.total); let capacityLabel = ''; let progressBar = html``; if (ev.total && ev.total > 0) { const percent = Math.min(100, Math.round((ev.booked / ev.total) * 100)); capacityLabel = `${this._formatNumber(ev.booked)} / ${this._formatNumber(ev.total)} kg`; progressBar = html`<div class="progress-track"><div class="progress-fill" style="width: ${percent}%;"></div></div>`; } return html`<div class="event-item ${isShort ? 'short-event' : ''} ${isFull ? 'fully-booked' : ''}" style="grid-row: ${startRow} / ${endRow};" title="${ev.title}"><div class="event-title">${ev.title}</div><div class="event-details-row">${isFull ? html`<span class="badge-booked-out">Ausgebucht</span>` : ''}${!isShort || !isFull ? html`<span class="capacity-text">${capacityLabel}</span>` : ''}${isShort && capacityLabel ? '' : html`<span style="margin-left:8px;">${ev.start} - ${ev.end}</span>`}</div>${progressBar}</div>`; }) : ''}<div class="grid-spacer"></div></div></div>`;
    }
}
customElements.define('winzer-day-view', WinzerDayView);
