import { LitElement, html, css, svg } from 'lit';
import { unsafeHTML } from 'lit/directives/unsafe-html.js';
import './card/winzer-file-card.js';
import './winzer-day-view.js';
import './winzer-menu-group.js';
import './winzer-modal.js';
import './card/winzer-card.js';
import './winzer-header.js';
import './winzer-nav.js';
import './toggle-switch.js';

export class WinzerApp extends LitElement {
    static properties = {
        isUserModalOpen: { type: Boolean },
        isFilterModalOpen: { type: Boolean },
        currentTab: { type: String },

        // Configurable properties
        appTitle: { type: String, attribute: 'app-title' },
        appSubtitle: { type: String, attribute: 'app-subtitle' },
        navItems: { type: Array }, // [{ id, label, icon (svg string), url }]
        menuAccount: { type: Array },
        menuSession: { type: Array },
    };

    constructor() {
        super();
        this.isUserModalOpen = false;
        this.isFilterModalOpen = false;
        this.currentTab = 'news'; // Default tab

        // Defaults
        this.appTitle = 'WINZER-PORTAL';
        this.appSubtitle = 'Badischer Winzerkeller';

        // Default Navigation (can be overwritten via property)
        this.navItems = [
            {
                id: 'news',
                label: 'News',
                url: '/news',
                icon: svg`<svg viewBox="0 0 24 24"><path d="M19 20H5a2 2 0 0 1-2-2V6a2 2 0 0 1 2-2h10a2 2 0 0 1 2-2v1m2 13a2 2 0 0 1-2-2V7m2 13a2 2 0 0 0 2-2V9a2 2 0 0 0-2-2h-2m-4-3H9M7 16h6M7 8h6v4H7V8z"/></svg>`
            },
            {
                id: 'anlieferung',
                label: 'Anlieferung',
                url: '/anlieferung',
                icon: svg`<svg viewBox="0 0 24 24"><circle cx="12" cy="10" r="2"/><circle cx="15.5" cy="8" r="2"/><circle cx="8.5" cy="8" r="2"/><circle cx="12" cy="6" r="2"/><circle cx="10" cy="13" r="2"/><circle cx="14" cy="13" r="2"/><circle cx="12" cy="16" r="2"/><path d="M12 2v4" stroke-linecap="round"/></svg>`
            },
            {
                id: 'tresor',
                label: 'Datei-Tresor',
                url: '/tresor',
                icon: svg`<svg viewBox="0 0 24 24"><path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"/><path d="M12 12v3"/><path d="M12 12v-1"/></svg>`
            }
        ];

        this.menuAccount = [
            { label: 'Persönliche Daten', icon: svg`<path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"/><circle cx="12" cy="7" r="4"/>`, action: 'profile' },
            { label: 'Sicherheits-Einstellungen', icon: svg`<rect x="3" y="11" width="18" height="11" rx="2" ry="2"/><path d="M7 11V7a5 5 0 0 1 10 0v4"/>`, action: 'security' },
            { label: 'Meine Buchungen', icon: svg`<circle cx="12" cy="10" r="2"/><circle cx="15.5" cy="8" r="2"/><circle cx="8.5" cy="8" r="2"/><circle cx="12" cy="6" r="2"/><circle cx="10" cy="13" r="2"/><circle cx="14" cy="13" r="2"/><circle cx="12" cy="16" r="2"/><path d="M12 2v4"/>`, action: 'bookings' }
        ];
        this.menuSession = [
            { label: 'Abmelden', icon: svg`<path d="M9 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h4"/><polyline points="16 17 21 12 16 7"/><line x1="21" y1="12" x2="9" y2="12"/>`, action: 'logout' }
        ];
    }

    static styles = css`
        :host {
            display: block;
            background-color: var(--bg-white);
            min-height: 100vh;
            width: 100%;
            position: relative;
        }
        .content { padding: 20px; padding-bottom: 90px; }
        .section-header { display: flex; align-items: center; gap: 12px; margin-bottom: 20px; }
        h1 { font-size: 1.5rem; font-weight: 800; margin: 0; text-transform: uppercase; }
        .abo-label { font-size: 0.8rem; font-weight: 600; margin-left: auto; margin-right: 8px; }

        /* Filter Bar */
        .filter-bar {
            display: flex;
            align-items: center;
            font-size: 0.9rem;
            font-weight: 600;
            margin-bottom: 16px;
            cursor: pointer;
        }
        .filter-icon {
            margin-left: 8px;
            width: 24px;
            height: 24px;
            color: var(--text-dark);
        }

        /* Filter Modal Styles */
        .filter-content {
            background-color: var(--bg-white);
            padding: 16px;
            border-radius: var(--radius-card);
            box-shadow: var(--shadow-sm);
        }
        .filter-row {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 12px;
        }
        .filter-row label {
            display: flex;
            align-items: center;
            font-weight: 700;
            font-size: 0.9rem;
            color: var(--text-dark);
        }
        .filter-row label svg {
            width: 20px;
            height: 20px;
            margin-right: 8px;
            color: var(--primary-color);
        }
        .filter-select {
            background-color: var(--bg-white);
            border: 1px solid var(--border-color);
            border-radius: 8px;
            padding: 8px 12px;
            font-size: 0.9rem;
            color: var(--text-dark);
            width: 160px;
        }
        .filter-btn {
            width: 100%;
            background-color: var(--primary-color);
            color: white;
            border: none;
            border-radius: var(--radius-sm);
            padding: 12px;
            font-weight: 600;
            font-size: 1rem;
            margin-top: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            cursor: pointer;
        }
    `;

    render() {
        return html`
            <!-- Header -->
            <winzer-header
                    .mainTitle="${this.appTitle}"
                    .subTitle="${this.appSubtitle}"
                    @user-click="${() => this.isUserModalOpen = true}">
            </winzer-header>

            <div class="content">
                <!-- Dynamic Content Slot -->
                <!--
                   The main content is now injected via the default slot.
                   Contao renders the page content, and it gets placed here.
                -->
                <slot></slot>
            </div>

            <!-- Bottom Nav -->
            <winzer-nav
                    .activeTab="${this.currentTab}"
                    .items="${this.navItems}">
            </winzer-nav>

            <!-- MODAL: USER -->
            <winzer-modal title="Mein Benutzerkonto" ?open="${this.isUserModalOpen}" @close="${() => this.isUserModalOpen = false}">
                <winzer-menu-group .items="${this.menuAccount}"></winzer-menu-group>
                <winzer-menu-group .items="${this.menuSession}"></winzer-menu-group>
            </winzer-modal>

            <!-- MODAL: FILTER -->
            <winzer-modal title="FILTER - EINSTELLUNGEN" ?open="${this.isFilterModalOpen}" @close="${() => this.isFilterModalOpen = false}">
                <div class="filter-content">
                    <div class="filter-row">
                        <label>
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="7" height="7"></rect><rect x="14" y="3" width="7" height="7"></rect><rect x="14" y="14" width="7" height="7"></rect><rect x="3" y="14" width="7" height="7"></rect></svg>
                            Kategorie:
                        </label>
                        <select class="filter-select">
                            <option>Traubengeld</option>
                            <option>Rechnungen</option>
                        </select>
                    </div>
                    <div class="filter-row">
                        <label>
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect><line x1="16" y1="2" x2="16" y2="6"></line><line x1="8" y1="2" x2="8" y2="6"></line><line x1="3" y1="10" x2="21" y2="10"></line></svg>
                            Zeitraum:
                        </label>
                        <select class="filter-select">
                            <option>Letzter Monat</option>
                            <option>Dieses Jahr</option>
                        </select>
                    </div>
                    <div class="filter-row" style="margin-top: 20px;">
                        <label>
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="21" y1="10" x2="3" y2="10"></line><line x1="21" y1="6" x2="3" y2="6"></line><line x1="21" y1="14" x2="3" y2="14"></line><line x1="21" y1="18" x2="3" y2="18"></line><polyline points="18 20 21 23 24 20"></polyline></svg>
                            Sortierung:
                        </label>
                        <select class="filter-select">
                            <option>Datum absteigend</option>
                            <option>Datum aufsteigend</option>
                        </select>
                    </div>

                    <button class="filter-btn" @click="${() => this.isFilterModalOpen = false}">
                        Filtern
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="5" y1="12" x2="19" y2="12"></line><polyline points="12 5 19 12 12 19"></polyline></svg>
                    </button>
                </div>
            </winzer-modal>
        `;
    }
}
customElements.define('winzer-app', WinzerApp);
