import { LitElement, html, css } from 'lit';

export class WinzerTag extends LitElement {
    static properties = {
        label: { type: String },
        color: { type: String }, // 'primary', 'secondary', 'success', 'danger', 'warning', 'info', 'light', 'dark'
        variant: { type: String }, // 'solid' (default), 'outline'
        size: { type: String }, // 'xs', 'sm' (default), 'md'
        removable: { type: Boolean },
        clickable: { type: Boolean }
    };

    static styles = css`
        :host {
            display: inline-flex;
            vertical-align: middle;
            line-height: 0;
        }
        .tag {
            display: flex;
            align-items: center;
            border-radius: 4px;
            font-weight: 600;
            line-height: normal;
            white-space: nowrap;
            border: 1px solid transparent;
            gap: 4px;
            transition: all 0.2s ease;
            box-sizing: border-box;
        }

        /* Sizes */
        .tag.xs { padding: 2px 6px; font-size: 0.65rem; border-radius: 3px; }
        .tag.sm { padding: 4px 8px; font-size: 0.75rem; }
        .tag.md { padding: 6px 10px; font-size: 0.85rem; }

        /* Clickable */
        .tag.clickable { cursor: pointer; }
        .tag.clickable:hover { opacity: 0.9; transform: translateY(-1px); }
        .tag.clickable:active { transform: translateY(0); }

        /* Solid Variants */
        .tag.solid { background-color: var(--bg-card); color: var(--text-main); border-color: var(--border-color); }
        .tag.solid.primary { background-color: var(--primary-light); color: var(--primary-color); border-color: var(--primary-color); }
        .tag.solid.secondary { background-color: var(--secondary-light); color: var(--secondary-color); border-color: var(--secondary-color); }
        .tag.solid.success { background-color: #d4edda; color: #155724; border-color: #c3e6cb; }
        .tag.solid.danger { background-color: #f8d7da; color: #721c24; border-color: #f5c6cb; }
        .tag.solid.warning { background-color: #fff3cd; color: #856404; border-color: #ffeeba; }
        .tag.solid.info { background-color: #d1ecf1; color: #0c5460; border-color: #bee5eb; }
        .tag.solid.light { background-color: #fefefe; color: #818182; border-color: #fdfdfe; }
        .tag.solid.dark { background-color: #d6d8d9; color: #1b1e21; border-color: #c6c8ca; }

        /* Outline Variants */
        .tag.outline { background-color: transparent; }
        .tag.outline.primary { color: var(--primary-color); border-color: var(--primary-color); }
        .tag.outline.secondary { color: var(--secondary-color); border-color: var(--secondary-color); }
        .tag.outline.success { color: #28a745; border-color: #28a745; }
        .tag.outline.danger { color: #dc3545; border-color: #dc3545; }
        .tag.outline.warning { color: #ffc107; border-color: #ffc107; }
        .tag.outline.info { color: #17a2b8; border-color: #17a2b8; }
        .tag.outline.light { color: #f8f9fa; border-color: #f8f9fa; }
        .tag.outline.dark { color: #343a40; border-color: #343a40; }

        /* Icon Slot Styling */
        ::slotted(svg) {
            width: 12px;
            height: 12px;
            fill: currentColor;
            display: block;
        }
        .tag.xs ::slotted(svg) {
            width: 10px;
            height: 10px;
        }
        .tag.md ::slotted(svg) {
            width: 14px;
            height: 14px;
        }

        /* Remove Button */
        .remove-btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin-left: 2px;
            cursor: pointer;
            opacity: 0.6;
            transition: opacity 0.2s;
            border-radius: 50%;
            width: 14px;
            height: 14px;
        }
        .tag.xs .remove-btn {
            width: 10px;
            height: 10px;
            margin-left: 1px;
        }
        .remove-btn:hover { opacity: 1; background-color: rgba(0,0,0,0.1); }
        .remove-btn svg { width: 10px; height: 10px; stroke-width: 3; }
        .tag.xs .remove-btn svg { width: 8px; height: 8px; }
    `;

    constructor() {
        super();
        this.variant = 'solid';
        this.size = 'sm';
        this.removable = false;
        this.clickable = false;
    }

    _handleRemove(e) {
        e.stopPropagation();
        this.dispatchEvent(new CustomEvent('remove', { bubbles: true, composed: true }));
    }

    render() {
        const classes = [
            'tag',
            this.variant || 'solid',
            this.size || 'sm',
            this.color || '',
            this.clickable ? 'clickable' : ''
        ].filter(Boolean).join(' ');

        return html`
            <div class="${classes}">
                <slot name="icon"></slot>
                <span>${this.label}</span>
                <slot></slot>
                ${this.removable ? html`
                    <span class="remove-btn" @click="${this._handleRemove}">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>
                    </span>
                ` : ''}
            </div>
        `;
    }
}
customElements.define('winzer-tag', WinzerTag);
