import { LitElement, html, css } from 'lit';

export class WinzerFileCard extends LitElement {
    static properties = {
        file: { type: Object } // { name, date, size, type, isNew }
    };

    static styles = css`
        :host {
            display: block;
            margin-bottom: 12px;
        }
        .file-card {
            background-color: var(--bg-card); /* Standard: Grau für gelesene Dateien */
            border-radius: var(--radius-sm);
            padding: 12px 16px;
            display: flex;
            align-items: center;
            border: 1px solid var(--border-color);
            box-shadow: var(--shadow-sm);
            position: relative;
            flex-wrap: wrap;
        }

        /* UPDATE: Styling für neue Dateien (Weißer BG + Farbiger Rand) */
        .file-card.has-new {
            background-color: var(--bg-white);
            border-color: var(--primary-color);
        }

        .icon-container {
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-right: 16px;
            flex-shrink: 0;
        }
        .icon-container svg {
            width: 32px;
            height: 32px;
        }
        .file-info {
            flex: 1;
            min-width: 0; /* Text truncation fix */
        }
        .file-name {
            font-weight: 400; /* UPDATE: Standard nicht fett */
            font-size: 0.9rem;
            color: var(--text-dark);
            margin-bottom: 4px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        /* UPDATE: Fett nur wenn neu */
        .has-new .file-name {
            font-weight: 700;
        }

        .file-meta-row {
            display: flex;
            align-items: center;
            gap: 12px;
            flex-wrap: wrap;
        }

        .file-meta {
            font-size: 0.75rem;
            color: var(--text-muted);
            line-height: 1; /* Ensure consistent height */
            display: flex;
            align-items: center;
        }
        .download-btn {
            width: 36px;
            height: 36px;
            border: 1px solid var(--primary-color);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;

            /* UPDATE: Standard Button Style (Weiß) */
            background-color: var(--bg-white);
            color: var(--primary-color);

            cursor: pointer;
            margin-left: 12px;
            flex-shrink: 0;

            /* UPDATE: Transition angepasst an WinzerCard */
            transition: transform 0.1s, background-color 0.2s;
        }

        /* UPDATE: Klick-Effekt hinzugefügt (wie bei WinzerCard) */
        .download-btn:active {
            transform: scale(0.95);
        }

        /* UPDATE: Button Style für neue Dateien (Primärfarbe) */
        .has-new .download-btn {
            background-color: var(--primary-color);
            color: white;
        }

        .download-btn svg {
            width: 20px;
            height: 20px;
        }

        /* New Indicator Dot */
        .new-dot {
            position: absolute;
            left: 6px; /* UPDATE: Position angepasst für fixe Ausrichtung */
            top: 50%;
            transform: translateY(-50%);
            width: 6px;
            height: 6px;
            background-color: var(--primary-color);
            border-radius: 50%;
        }
        /* UPDATE: Padding-Shift entfernt, damit Icons ausgerichtet bleiben */

        /* Type Colors */
        .type-pdf { color: var(--alert-color); }
        .type-doc { color: #2b5797; } /* Word blueish */
        .type-default { color: var(--text-muted); }

        /* Slot for tags */
        .tags-container {
            display: flex;
            align-items: center; /* Ensure tags are vertically centered */
            gap: 8px;
            flex-wrap: wrap;
        }
    `;

    _getIcon(type) {
        const t = type.toLowerCase();
        if (t === 'pdf') {
            return html`<svg class="type-pdf" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path><polyline points="14 2 14 8 20 8"></polyline><path d="M9 15l3 3 3-3"></path><path d="M12 18v-6"></path><text x="6" y="22" font-size="6" font-weight="bold" fill="currentColor" stroke="none">PDF</text></svg>`;
        } else if (t === 'docx' || t === 'doc') {
            return html`<svg class="type-doc" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"><path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path><polyline points="14 2 14 8 20 8"></polyline><line x1="16" y1="13" x2="8" y2="13"></line><line x1="16" y1="17" x2="8" y2="17"></line><line x1="10" y1="9" x2="8" y2="9"></line></svg>`;
        }
        return html`<svg class="type-default" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M13 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V9z"></path><polyline points="13 2 13 9 20 9"></polyline></svg>`;
    }

    render() {
        return html`
            <div class="file-card ${this.file.isNew ? 'has-new' : ''}">
                ${this.file.isNew ? html`<div class="new-dot"></div>` : ''}

                <div class="icon-container">
                    ${this._getIcon(this.file.type)}
                </div>

                <div class="file-info">
                    <div class="file-name">${this.file.name}</div>
                    <div class="file-meta-row">
                        <div class="file-meta">${this.file.date} &nbsp; ${this.file.size}</div>
                        <div class="tags-container">
                            <slot name="tags"></slot>
                        </div>
                    </div>
                </div>

                <div class="download-btn">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="7 10 12 15 17 10"></polyline><line x1="12" y1="15" x2="12" y2="3"></line></svg>
                </div>
            </div>
        `;
    }
}
customElements.define('winzer-file-card', WinzerFileCard);
